/**
 * collectd - src/daemon/metric.h
 * Copyright (C) 2019-2020  Google LLC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *   Florian octo Forster <octo at collectd.org>
 *   Manoj Srivastava <srivasta at google.com>
 **/

#ifndef METRIC_H
#define METRIC_H 1

#include "utils/metadata/meta_data.h"
#include "utils/strbuf/strbuf.h"
#include "utils_time.h"

#define METRIC_ATTR_DOUBLE 0x01
#define METRIC_ATTR_CUMULATIVE 0x02
#define METRIC_ATTR_MONOTONIC 0x04

typedef enum {
  METRIC_TYPE_UNTYPED = 0,
  // METRIC_TYPE_GAUGE are absolute metrics that cannot (meaningfully) be summed
  // up. Examples are temperatures and utilization ratios.
  METRIC_TYPE_GAUGE = METRIC_ATTR_DOUBLE,
  // METRIC_TYPE_COUNTER are monotonically increasing integer counts. The rate
  // of change is meaningful, the absolute value is not.
  METRIC_TYPE_COUNTER = METRIC_ATTR_CUMULATIVE | METRIC_ATTR_MONOTONIC,
  // METRIC_TYPE_COUNTER_FP are monotonically increasing floating point counts.
  // The rate of change is meaningful, the absolute value is not.
  METRIC_TYPE_COUNTER_FP =
      METRIC_ATTR_DOUBLE | METRIC_ATTR_CUMULATIVE | METRIC_ATTR_MONOTONIC,
  // METRIC_TYPE_UP_DOWN are absolute integer metrics that can
  // (meaningfully) be summed up. Examples are filesystem space used and
  // physical memory.
  METRIC_TYPE_UP_DOWN = METRIC_ATTR_CUMULATIVE,
  // METRIC_TYPE_UP_DOWN_FP are absolute floating point metrics that can
  // (meaningfully) be summed up.
  METRIC_TYPE_UP_DOWN_FP = METRIC_ATTR_DOUBLE | METRIC_ATTR_CUMULATIVE,
} metric_type_t;

#define METRIC_TYPE_TO_STRING(t)                                               \
  (t == METRIC_TYPE_GAUGE)        ? "gauge"                                    \
  : (t == METRIC_TYPE_COUNTER)    ? "counter"                                  \
  : (t == METRIC_TYPE_COUNTER_FP) ? "counter_fp"                               \
  : (t == METRIC_TYPE_UP_DOWN)    ? "up_down"                                  \
  : (t == METRIC_TYPE_UP_DOWN_FP) ? "up_down_fp"                               \
                                  : "unknown"

#define IS_DOUBLE(t) ((t)&METRIC_ATTR_DOUBLE)
#define IS_MONOTONIC(t) ((t)&METRIC_ATTR_MONOTONIC)

typedef double gauge_t;
typedef uint64_t counter_t;
typedef int64_t derive_t;

union value_u {
  gauge_t gauge;
  counter_t counter;
  double counter_fp;
  int64_t up_down;
  double up_down_fp;
  // For collectd 5 compatiblity. Treated the same as up_down.
  derive_t derive;
};
typedef union value_u value_t;

/* value_marshal_text prints a text representation of v to buf. */
int value_marshal_text(strbuf_t *buf, value_t v, metric_type_t type);

/*
 * Labels
 */
/* label_pair_t represents a label, i.e. a key/value pair. */
typedef struct {
  char *name;
  char *value;
} label_pair_t;

/* label_set_t is a sorted set of labels. */
typedef struct {
  label_pair_t *ptr;
  size_t num;
} label_set_t;

/* label_set_clone copies all the labels in src into dest. dest must be an empty
 * label set, i.e. it must not contain any prior labels, otherwise EINVAL is
 * returned. */
int label_set_clone(label_set_t *dest, label_set_t src);

/* label_set_get looks up the label pair by name, and returns the associated
 * value. Returns NULL and sets errno to ENOENT if the label doesn't exist. */
char const *label_set_get(label_set_t labels, char const *name);

/* label_set_add adds a label to the label set. If a label with name already
 * exists, EEXIST is returned. The set of labels is sorted by label name. */
int label_set_add(label_set_t *labels, char const *name, char const *value);

/* label_set_update adds, updates, or deletes a label pair. If "value" is NULL
 * or an empty string, the label is removed.
 * Removing a label that does not exist is *not* an error. */
int label_set_update(label_set_t *labels, char const *name, char const *value);

/* label_set_reset frees all the memory referenced by the label set and
 * initializes the label set to zero. */
void label_set_reset(label_set_t *labels);

/* label_set_compare compares two label sets. It returns an integer indicating
 * the result of the comparison, as follows:
 *
 *   - 0, if the a and b are equal;
 *   - a negative value if a is less than b;
 *   - a positive value if a is greater than b.
 */
int label_set_compare(label_set_t a, label_set_t b);

/* label_set_format formats a label set as a string, returned in "buf".
 * Returns zero on success and non-zero on failure. */
int label_set_format(strbuf_t *buf, label_set_t labels);

/*
 * Metric
 */
/* forward declaration since metric_family_t and metric_t refer to each other.
 */
struct metric_family_s;
typedef struct metric_family_s metric_family_t;

/* metric_t is a metric inside a metric family. */
typedef struct {
  metric_family_t *family; /* backreference for family->name and family->type */

  label_set_t label;

  value_t value;
  cdtime_t time; /* TODO(octo): use ms or µs instead? */
  cdtime_t interval;
  meta_data_t *meta;
} metric_t;

/* metric_identity writes the identity of the metric "m" to "buf", using the
 * OpenMetrics / Prometheus plain text exposition format.
 *
 * Example:
 *   "http_requests_total{method=\"post\",code=\"200\"}"
 */
int metric_identity(strbuf_t *buf, metric_t const *m);

/* metric_parse_identity parses "s" and returns a metric with only its identity
 * set. On error, errno is set and NULL is returned. The returned memory must
 * be freed by passing m->family to metric_family_free(). */
metric_t *metric_parse_identity(char const *s);

/* metric_label_set adds or updates a label to/in the label set.
 * If "value" is NULL or the empty string, the label is removed. Removing a
 * label that does not exist is *not* an error. */
int metric_label_set(metric_t *m, char const *name, char const *value);

/* metric_label_get efficiently looks up and returns the value of the "name"
 * label. If a label does not exist, NULL is returned and errno is set to
 * ENOENT. The returned pointer may not be valid after a subsequent call to
 * "metric_label_set". */
char const *metric_label_get(metric_t const *m, char const *name);

/* metric_reset frees all labels and meta data stored in the metric and resets
 * the metric to zero. */
int metric_reset(metric_t *m);

/* metric_list_t is an unordered list of metrics. */
typedef struct {
  metric_t *ptr;
  size_t num;
} metric_list_t;

/*
 * Metric Family
 */
/* metric_family_t is a group of metrics of the same type. */
struct metric_family_s {
  char *name;
  char *help;
  /* unit is a case sensitive "Unified Code for Units of Measure" (UCUM)
   * denoting the unit of the metric, e.g. "By" for bytes, and "1" for
   * dimensionless metrics, such as "utilization". */
  char *unit;
  metric_type_t type;

  label_set_t resource;
  metric_list_t metric;
};

/* metric_family_append_list appends a metric_list_t to the metric family. */
int metric_family_append_list(metric_family_t *fam, metric_list_t list);

/* metric_family_metric_append appends a new metric to the metric family. This
 * allocates memory which must be freed using metric_family_metric_reset. */
int metric_family_metric_append(metric_family_t *fam, metric_t m);

/* metric_family_resource_attribute_update adds, updates, or deletes a
 * resource attribute. If "value" is NULL or an empty string, the attribute
 * is removed.
 * Removing an attribute that does not exist is *not* an error. */
int metric_family_resource_attribute_update(metric_family_t *fam,
                                            char const *name,
                                            char const *value);

/* metric_family_append constructs a new metric_t and appends it to fam. It is
 * a convenience function that is funcitonally approximately equivalent to the
 * following code, but without modifying templ:
 *
 *   metric_t m = *templ;
 *   m.value = v;
 *   metric_label_set(&m, lname, lvalue);
 *   metric_family_metric_append(fam, m);
 *
 * templ may be NULL. In that case the function behaves as if &(metric_t){0} had
 * been passed in.
 */
int metric_family_append(metric_family_t *fam, char const *lname,
                         char const *lvalue, value_t v, metric_t const *templ);

/* metric_family_metric_reset frees all metrics in the metric family and
 * resets the count to zero. */
int metric_family_metric_reset(metric_family_t *fam);

/* metric_family_free frees a "metric_family_t" that was allocated with
 * metric_family_clone(). */
void metric_family_free(metric_family_t *fam);

/* metric_family_clone returns a copy of the provided metric family. On error,
 * errno is set and NULL is returned. The returned pointer must be freed with
 * metric_family_free(). */
metric_family_t *metric_family_clone(metric_family_t const *fam);

/* metric_family_clone_shallow returns a copy of the provided metric family
 * without any metrics. On error, errno is set and NULL is returned. The
 * returned pointer must be freed with metric_family_free(). */
metric_family_t *metric_family_clone_shallow(metric_family_t const *fam);

/* metric_family_compare compares two metric families, taking into account the
 * metric family name and any resource attributes. It returns an integer
 * indicating the result of the comparison, as follows:
 *
 *   - 0, if the a and b are equal;
 *   - a negative value if a is less than b;
 *   - a positive value if a is greater than b.
 */
int metric_family_compare(metric_family_t const *a, metric_family_t const *b);

#endif
